package com.pdftools.pdfvalidator;

import com.pdftools.NativeLibrary;
import com.pdftools.Stream;

/**
 * The purpose of the PDF validator API is to provide the functionality of validating PDF files for
   conformance with various standards. The level of compliance can be set upon opening a PDF file.
 */
public class PdfValidatorAPI extends NativeLibrary
{
    /**
     * Set the license key.
     * @param licenseKey The license key
     * @return True if the license is valid.
     */
    public static native boolean setLicenseKey(String licenseKey);

    /**
     * Check if a valid license is loaded.
     * @return True if a valid license is loaded.
     */
    public static native boolean getLicenseIsValid();

     /**
      * Get the version of the product
      * @return a string in the format 'A.B.C.D'.
      */
     public static native String getProductVersion(); 

    /**
     * Terminate and unload drivers of all cryptograpic providers.
     * <p>
     * Calling Terminate is mandatory, if in the custom validator profile a 
     * PKCS#11 device is configured for signature validation (key Provider).
     * <p>
     * Some drivers require Terminate to be called. Otherwise, your application
     * might crash and/or your HSM, USB token or smart card might not be
     * unlocked. Make sure to dispose of all PdfValidatorAPI objects before
     * calling Terminate. After calling Terminate, the process may not call
     * any other methods of this class.
     */
    public static native void terminate();

    /**
     * default constructor
     */
    public PdfValidatorAPI()
    {
        createObject();
    }

    /**
     * Set handle for the process.
     */
    private native boolean createObject();

	/**
    * Release handle for the process.
    */
	public native void destroyObject();

	/**
	 * Open an input PDF file for validation. This makes the objects contained
	 * in the PDF document accessible. If the document is already open, it is
	 * closed first.
	 * 
	 * @param strFileName
	 *            The source file name and the file path, drive or server string
	 *            according to the operating systems file name specification
	 *            rules of the input file.
	 *            <p>
	 *            Path: e.g. c:\data\document.pdf <br/>
	 *            HTTP URL: http://[username:password@]domain[:port][/resource],
	 *            where username:pass-word are used for HTTP basic
	 *            authentication. <br/>
	 *            HTTPS URL: URL beginning with https:// <br/>
	 *            FTP URL: URL beginning with ftp:// <br/>
	 * @param strPassword
	 *            The user or the owner password of the encrypted input PDF
	 *            document. If this parameter is left out, null or an empty
	 *            string the default (no password) is used.
	 * @param iCompliance
	 *            The compliance level: See enumeration
	 *            {@link com.pdftools.NativeLibrary.COMPLIANCE}. If ePDFUnk is
	 *            passed, the Validator determines the claimed compliance of the
	 *            document. <br/>
	 *            The determined compliance can be read using the property
	 *            {@link #getCompliance getCompliance} and will be used in the
	 *            {@link #validate validate} method. <br/>
	 *            Note that the claimed compliance is not limited to a version
	 *            of PDF/A.
	 * @return True: The file could be successfully opened. False: The file does
	 *         not exist, is corrupt, or the password is not valid.
	 */
    public native boolean open(String strFileName, String strPassword, int iCompliance);

	/**
	 * Open an input PDF file memory block for validation. This makes the objects
	 * contained in the PDF document accessible. If the document is already open
	 * it is closed first.
	 * 
	 * @param data
	 *            Memory block containing the input PDF file as one dimensional
	 *            byte array.
	 * @param strPassword
	 *            The user or the owner password of the encrypted input PDF
	 *            document. If this parameter is left out, null or an empty
	 *            string the default (no password) is used.
	 * @param iCompliance
	 *            The compliance level: See enumeration
	 *            {@link com.pdftools.NativeLibrary.COMPLIANCE}. If ePDFUnk is
	 *            passed, the Validator determines the claimed compliance of the
	 *            document. <br/>
	 *            The determined compliance can be read using the property
	 *            {@link #getCompliance getCompliance} and will be used in the
	 *            {@link #validate validate} method. <br/>
	 *            Note that the claimed compliance is not limited to a version
	 *            of PDF/A.
	 * @return True: The file could be successfully opened. False: The file does
	 *         not exist, is corrupt, or the password is not valid.
	 */
	public native boolean open(byte[] data, String strPassword, int iCompliance);

	/**
	 * Open an input stream for validation.
	 * 
	 * @param stream
	 *            The input PDF as a stream.
	 * @param strPassword
	 *            The user or the owner password of the encrypted input PDF
	 *            document. If this parameter is left out, null or an empty
	 *            string the default (no password) is used.
	 * @param iCompliance
	 *            The compliance level: See enumeration
	 *            {@link com.pdftools.NativeLibrary.COMPLIANCE}. If ePDFUnk is
	 *            passed, the Validator determines the claimed compliance of the
	 *            document. <br/>
	 *            The determined compliance can be read using the property
	 *            {@link #getCompliance getCompliance} and will be used in the
	 *            {@link #validate validate} method. <br/>
	 *            Note that the claimed compliance is not limited to a version
	 *            of PDF/A.
	 * @return True: The file could be successfully opened. False: The file does
	 *         not exist, is corrupt, or the password is not valid.
	 */
    public native boolean open(Stream stream, String strPassword, int iCompliance);

	/**
	 * Closes the currently open input file(s). If the document is already closed
	 * the method does nothing.
	 * 
	 * @return True: The input file was closed successfully. False otherwise.
	 */
    public native boolean close();

	/**
	 * Starts the validation of the document and prepares for error retrieval
	 * through {@link #getFirstError getFirstError} and {@link #getNextError
	 * getNextError}. It aborts on the first error if {@link #setStopOnError
	 * setStopOnError} is set to true.
	 * <p>
	 * The return value alone does not give an indication whether the document
	 * is compliant or not. The document is compliant, if and only if, validate
	 * returns True and the {@link #getErrorCode getErrorCode} is not set to
	 * PDFErrorCode.
	 * 
	 * @return True if the validation finished successfully and False if the
	 *         validation was aborted (e.g. because an error was found and
	 *         setStopOnError is set to True).
	 */
    public native boolean validate();

    /**
    * Gets the first error message from the list, it can also be a warning.
    * @return The first PDFError if there is any. Nothing otherwise.
    */
    public native PdfError getFirstError();

    /**
    * Gets the next error message from the list, it can also be a warning.
    * @return The next PDFError if there is any. Nothing otherwise.
    */
    public native PdfError getNextError();

    /**
    * Gets the latest error code.
    * @see com.pdftools.NativeLibrary.ERRORCODE
    * @return Latest error code
    */
    public native int getErrorCode();

	/**
	 * Gets the message describing the last error that occurred while
	 * processing. The property is NULL if no message is available.
	 * 
	 * @return Latest error message
	 */
    public native String getErrorMessage();

	/**
	 * Defines whether to abort validation on the first error. If set to true,
	 * the method {@link #validate validate} will abort on the first validation
	 * error (but not on warnings), i.e. the validation process will stop
	 * (errorPDF_E_STOPPED) as soon as a problem is found that makes the file
	 * non-conforming. This speeds up the validation of non-conforming files.
	 * This property must always be set after {@link #open open}. It is set to
	 * False after a call to {@link #close close}. Default: False.
	 * 
	 * @param bVal
	 *            If set to true, the method validate will abort on the first
	 *            error. However, it will continue on warnings and information
	 *            messages until the first error occurs.
	 */
    public native void setStopOnError(boolean bVal);

	/**
	 * Gets whether validation is aborted on the first error. If set to true,
	 * the method {@link #validate validate} will abort on the first validation
	 * error (but not on warnings), i.e. the validation process will stop
	 * (errorPDF_E_STOPPED) as soon as a problem is found that makes the file
	 * non-conforming. This speeds up the validation of non-conforming files.
	 * This property must always be set after {@link #open open}. It is set to
	 * False after a call to {@link #close close}. Default: False.
	 * 
	 * @return If true, the method validate will abort on the first error.
	 *         However, it will continue on warnings and information messages
	 *         until the first error occurs.
	 */
    public native boolean getStopOnError();

	/**
	 * Defines which types of messages are returned. Only error messages are
	 * logged that have an error level that is equal or higher to this property.
	 * Supported levels are:
	 * <p>
	 * 0 none - Nothing is reported. <br/>
	 * 1 errors - Errors only are reported. <br/>
	 * 2 warnings - Errors and warnings are reported. <br/>
	 * 3 information - Errors, warnings and information messages are reported.
	 * <br/>
	 * This property must be set before the {@link #open open} method in order
	 * to be applied. All errors are listed in the header file bseerror.h.
	 * 
	 * @param iVal
	 *            Reporting level
	 */
    public native void setReportingLevel(int iVal);

	/**
	 * Gets which types of messages are returned. Only error messages are logged
	 * that have an error level that is equal or higher to this property.
	 * Supported levels are:
	 * <p>
	 * 0 none - Nothing is reported <br/>
	 * 1 errors - Errors only are reported <br/>
	 * 2 warnings - Errors and warnings are reported <br/>
	 * 3 information - Errors, warnings and information messages are reported
	 * 
	 * @return Actual reporting level
	 */
    public native int getReportingLevel();

	/**
	 * Report summary as bitmask of error categories. Instead of a detailed
	 * report using {@link #getFirstError getFirstError} and
	 * {@link #getNextError getNextError} there is an alternative to report a
	 * summary. The summary consists of 19 possible messages (see property
	 * {@link #getCategoryText getCategoryText}. If any violation is detected at
	 * least once, it is reported once. The property getCategories can be got after
	 * the validation. It returns a number in which each bit represents one of
	 * these 19 messages. The textual value for each bit can be retrieved using
	 * getCategoryText(Bit).
	 */
    public native int getCategories();

	/**
	 * Return a textual description for each of the 19 summary messages (see
	 * property {@link #getCategories getCategories}). The messages are described in
	 * {@link com.pdftools.NativeLibrary.PDFCONFORMANCECATEGORY}.
	 */
    public native String getCategoryText(int iCategory);

	/**
	 * Gets the compliance used for validation of the currently open document.
	 * This is usually the same value as provided in the {@link #open open}
	 * method (unless ePDFUnk was supplied). This property must be read after
	 * {@link #open open}. It is no longer meaningful after a call to
	 * {@link #close close}.
	 */
    public native int getCompliance();

	/**
	 * Defines whether the Validator creates temporary files. If set to true, the
	 * Validator will not create any temporary files. If set to False, temporary
	 * files might be created, e.g. for embedded files. Use this option with
	 * care, because if set to True this might increase memory consumption
	 * significantly.
	 */
    public native void setNoTempFiles(boolean bVal);

	/**
	 * Gets whether the Validator creates temporary files.
	 * 
	 * @return If true, the Validator will not create any temporary files. If
	 *         False, temporary files might be created, e.g. for embedded files.
	 */
    public native boolean getNoTempFiles();

    /**
    * Sets custom validation profile to validate compliance with corporate directives.
    * @param fileName The file name of the profile configuration file. Set fileName to null or the empty string, in order to remove the active profile.
    * @return True if the profile was set successfully.
    */
    public native boolean setProfile(String fileName);

    /**
    * Sets custom validation profile to validate compliance with corporate directives.
    * @param data The profile configuration file as byte array. Set data to null in order to remove the active profile.
    * @return True if the profile was set successfully.
    */
    public native boolean setProfile(byte[] data);
    
    /**
    * Sets custom validation profile to validate compliance with corporate directives.
    * @param stream The profile configuration file as stream. Set stream to null in order to remove the active profile.
    * @return True if the profile was set successfully.
    */
    public native boolean setProfile(Stream stream);

	/**
	 * Write font validation data to stream.
	 * 
	 * @param stream
	 *            The stream the XML is written to.
	 * @return True: Font validation has successuflly been written. False:otherwise.
	 */
    public native boolean writeFontValidationXML(Stream stream);

    protected static native void initialize(String strVersion);

    protected long _handle;

    /** load the native library */
    static 
    {
        try 
        {
            initialize(VERSION);
        } 
        catch (UnsatisfiedLinkError ue) 
        {
            System.loadLibrary("PdfValidatorAPI");
            initialize(VERSION);
        }
    }

    /**
    * The standard finalize method frees internal native resources
    * in case the destroyObject() call is missing in the application.
    */
	protected void finalize() throws Throwable
	{
		destroyObject();
	}
}
